<?php
use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

require __DIR__ . '/phpmailer/autoload.php';
require __DIR__ . '/tcpdf/tcpdf.php';

header('Content-Type: application/json');
session_start();

error_reporting(E_ALL);
ini_set('display_errors', 0);

// force JSON output on error
set_exception_handler(function ($e) {
    echo json_encode(['error' => 'Exception: ' . $e->getMessage()]);
    exit;
});
set_error_handler(function ($severity, $message, $file, $line) {
    echo json_encode(['error' => "PHP Error: $message in $file line $line"]);
    exit;
});

// Ensure generated_pdfs folder exists and is writable
$generated_dir = __DIR__ . '/generated_pdfs';
if (!is_dir($generated_dir)) {
    @mkdir($generated_dir, 0755, true);
}

// Read action (support JSON body too)
$action = $_POST['action'] ?? null;
if (!$action) {
    $raw = file_get_contents('php://input');
    $json = @json_decode($raw, true);
    if (json_last_error() === JSON_ERROR_NONE && isset($json['action'])) {
        foreach ($json as $k => $v) { if (!isset($_POST[$k])) $_POST[$k] = $v; }
        $action = $json['action'];
    }
}

$action = $action ?? '';

switch ($action) {
    case 'save_config':
        $_SESSION['mailer_config'] = [
            'from_name'    => $_POST['from_name'] ?? '',
            'from_email'   => $_POST['from_email'] ?? '',
            'bounce_email' => $_POST['bounce_email'] ?? '',
            'subject'      => $_POST['subject'] ?? '',
            'message'      => $_POST['message'] ?? '',
            'email_list'   => preg_split('/\r\n|\r|\n/', trim($_POST['email_list'] ?? '')),
            // optional SMTP
            'smtp_host'    => $_POST['smtp_host'] ?? '',
            'smtp_user'    => $_POST['smtp_user'] ?? '',
            'smtp_pass'    => $_POST['smtp_pass'] ?? '',
            'smtp_port'    => $_POST['smtp_port'] ?? '',
            'smtp_secure'  => $_POST['smtp_secure'] ?? '', // tls, ssl, none
            // pdf generation
            'filename_template' => $_POST['filename_template'] ?? 'document_{email}.pdf',
            'pdf_html' => $_POST['pdf_html'] ?? '',
            'extra_html' => $_POST['extra_html'] ?? ''
        ];
        $_SESSION['status'] = [
            'status'=>'idle',
            'queue_length'=>count($_SESSION['mailer_config']['email_list']),
            'current_index'=>0,
            'sent_count'=>0,
            'failed_count'=>0,
            'sent_emails'=>[],
            'failed_emails'=>[]
        ];
        echo json_encode($_SESSION['status']);
        break;

    case 'get_status':
        echo json_encode($_SESSION['status'] ?? []);
        break;

    case 'start_sending':
        if (!isset($_SESSION['status'])) {
            echo json_encode(['error'=>'No configuration saved.']);
            break;
        }
        $_SESSION['status']['status'] = 'running';
        echo json_encode($_SESSION['status']);
        break;

    case 'process_batch':
        $cfg = $_SESSION['mailer_config'] ?? [];
        $st  =& $_SESSION['status'];
        $emails = $cfg['email_list'] ?? [];

        $processed = 0;

        // safety
        if (!is_array($emails)) $emails = [];

        // send one email per batch
        if ($st['current_index'] < count($emails)) {
            $recipient = trim($emails[$st['current_index']]);
            if ($recipient) {
                // prepare subject and body with tags
                $subjectWithTags = applyTags($cfg['subject'] ?? '', $recipient, $cfg);
                $messageWithTags = applyTags($cfg['message'] ?? '', $recipient, $cfg);

                // generate personalized PDF
                $filenameTemplate = $cfg['filename_template'] ?? 'document_{email}.pdf';
                $outputFilename = applyTags($filenameTemplate, $recipient, $cfg);
                // sanitize filename
                $outputFilename = preg_replace('/[^A-Za-z0-9_\-\.]/', '_', $outputFilename);
                $outputPath = $GLOBALS['generated_dir'] . '/' . $outputFilename;

                $html = $cfg['pdf_html'] ?? '';
                $extra = $cfg['extra_html'] ?? '';
                $fullHtml = applyTags($html, $recipient, $cfg) . '<br>' . applyTags($extra, $recipient, $cfg);

                $genOk = generatePdfFromHtml($fullHtml, $outputPath);
                if (!$genOk) {
                    // failed generating PDF
                    $st['failed_count']++;
                    $st['failed_emails'][$recipient] = ['error'=>'PDF generation failed','timestamp'=>time()];
                } else {
                    $ok = sendEmail(
                        $cfg['from_email'] ?? ('no-reply@' . ($_SERVER['SERVER_NAME'] ?? 'localhost')),
                        $cfg['from_name'] ?? '',
                        $recipient,
                        $subjectWithTags,
                        $messageWithTags,
                        $cfg,
                        $outputPath
                    );

                    if ($ok) {
                        $st['sent_count']++;
                        $st['sent_emails'][] = [
                            'email'=>$recipient,
                            'subject'=>$subjectWithTags,
                            'pdf'=>$outputFilename,
                            'timestamp'=>time()
                        ];
                    } else {
                        $st['failed_count']++;
                        $st['failed_emails'][$recipient] = [
                            'subject'=>$subjectWithTags,
                            'attempts'=>($st['failed_emails'][$recipient]['attempts']??0)+1,
                            'last_attempt'=>time()
                        ];
                    }
                }
                $processed++;
            }
            $st['current_index']++;

            // enforce delay (configurable via emails_per_second or delay_seconds)
            $delay = floatval($_POST['delay_seconds'] ?? ($_POST['emails_per_second'] ? 1.0 / floatval($_POST['emails_per_second']) : 3.6));
            if ($delay <= 0) $delay = 3.6;
            // sleep accepts integer seconds; use usleep for fractions
            usleep(intval($delay * 1000000));
        }

        if ($st['current_index'] >= count($emails)) {
            $st['status'] = 'idle';
        }

        $st['processed'] = $processed;
        echo json_encode($st);
        break;

    case 'pause_sending':
        $_SESSION['status']['status'] = 'paused';
        echo json_encode($_SESSION['status']);
        break;

    case 'resume_sending':
        $_SESSION['status']['status'] = 'running';
        echo json_encode($_SESSION['status']);
        break;

    case 'stop_sending':
        $_SESSION['status']['status'] = 'idle';
        echo json_encode($_SESSION['status']);
        break;

    case 'reset_counters':
        $_SESSION['status']['sent_count'] = 0;
        $_SESSION['status']['failed_count'] = 0;
        $_SESSION['status']['sent_emails'] = [];
        $_SESSION['status']['failed_emails'] = [];
        $_SESSION['status']['current_index'] = 0;
        echo json_encode($_SESSION['status']);
        break;

    case 'retry_failed':
        $retry = array_keys($_SESSION['status']['failed_emails'] ?? []);
        $_SESSION['mailer_config']['email_list'] = array_merge($_SESSION['mailer_config']['email_list'] ?? [], $retry);
        $_SESSION['status']['queue_length'] = count($_SESSION['mailer_config']['email_list']);
        $_SESSION['status']['failed_emails'] = [];
        echo json_encode($_SESSION['status']);
        break;

    case 'preview':
        // generate a single PDF and return base64 for preview
        $cfg = $_SESSION['mailer_config'] ?? [];
        $recipient = $_POST['recipient'] ?? ($_POST['preview_email'] ?? 'preview@example.com');
        $html = applyTags($cfg['pdf_html'] ?? '', $recipient, $cfg) . '<br>' . applyTags($cfg['extra_html'] ?? '', $recipient, $cfg);
        $tmp = tempnam(sys_get_temp_dir(), 'pdfpreview_') . '.pdf';
        $ok = generatePdfFromHtml($html, $tmp);
        if ($ok) {
            $b = base64_encode(file_get_contents($tmp));
            @unlink($tmp);
            echo json_encode(['success'=>true,'pdf_base64'=>$b]);
        } else {
            echo json_encode(['success'=>false,'error'=>'PDF generation failed']);
        }
        break;

    default:
        echo json_encode(['error'=>'Unknown action']);
}

// ========= Email Sender ==========
function sendEmail($from,$name,$to,$subj,$body,$cfg=[],$attachment=null){
    $mail = new PHPMailer(true);
    try {
        if (!empty($cfg['smtp_host']) && !empty($cfg['smtp_user'])) {
            $mail->isSMTP();
            $mail->Host = $cfg['smtp_host'];
            $mail->SMTPAuth = true;
            $mail->Username = $cfg['smtp_user'];
            $mail->Password = $cfg['smtp_pass'];
            $mail->Port     = $cfg['smtp_port'] ?: 587;
            if (!empty($cfg['smtp_secure']) && $cfg['smtp_secure']!=='none') {
                $mail->SMTPSecure = ($cfg['smtp_secure']==='ssl')
                    ? PHPMailer::ENCRYPTION_SMTPS
                    : PHPMailer::ENCRYPTION_STARTTLS;
            }
        } else {
            $mail->isMail();
        }

        $mail->setFrom($from,$name);
        $mail->addAddress($to);
        if (!empty($cfg['bounce_email'])) $mail->Sender=$cfg['bounce_email'];

        $mail->CharSet  = 'UTF-8';
        $mail->Encoding = 'base64';

        $mail->isHTML(true);
        $mail->Subject = $subj;
        $mail->Body    = $body;
        $mail->AltBody = strip_tags(preg_replace('/<br\s*\/?>/i', "\\n", $body));

        if ($attachment && file_exists($attachment)) {
            $mail->addAttachment($attachment);
        }

        $mail->send();
        return true;
    } catch (Exception $e) {
        error_log("Mailer Error to {$to}: ".$mail->ErrorInfo);
        return false;
    }
}

// ========= PDF Generation ==========
function generatePdfFromHtml($html, $outputPath) {
    try {
        // Create new TCPDF instance
        $pdf = new TCPDF(PDF_PAGE_ORIENTATION, PDF_UNIT, PDF_PAGE_FORMAT, true, 'UTF-8', false);
        $pdf->SetCreator(PDF_CREATOR);
        $pdf->SetAuthor('Mailer');
        $pdf->SetTitle('Document');
        $pdf->SetSubject('Generated PDF');
        $pdf->SetMargins(15, 15, 15);
        $pdf->SetAutoPageBreak(TRUE, 15);
        $pdf->setFontSubsetting(true);
        $pdf->AddPage();
        $pdf->writeHTML($html, true, false, true, false, '');
        $pdf->lastPage();
        $pdf->Output($outputPath, 'F');
        return file_exists($outputPath);
    } catch (Exception $e) {
        error_log('PDF generation error: '.$e->getMessage());
        return false;
    }
}

// ========= Tag Engine ==========
function applyTags($text, $recipient, $config) {
    $parts = explode('@', $recipient);
    $username = $parts[0] ?? '';
    $domain   = $parts[1] ?? '';

    $usernameParts = preg_split('/[._]/', $username);
    $firstName = ucfirst($usernameParts[0] ?? '');
    $lastName  = ucfirst(end($usernameParts));

    $tags = [
        '{email}'        => $recipient,
        '{username}'     => $username,
        '{domain}'       => $domain,
        '{first_name}'   => $firstName,
        '{last_name}'    => $lastName,
        '{from_name}'    => $config['from_name'] ?? '',
        '{timestamp}'    => time(),
        '{date}'         => date('Y-m-d'),
        '{base64:email}' => base64_encode($recipient),
    ];

    return strtr($text, $tags);
}
?>
